"""symfer I/O functions for Hugin (.net), YAML and C files"""

import yaml
import os
import tempfile
import distutils.ccompiler
import subprocess
from . import huginparser
from .utils import normalize
from .factor import *
from . import cgenerator

__all__ = """
    loadhugin
    loadyaml   dumpyaml
    dumpc   evalc
    """.split()

def loadhugin(filename='tmp.net',uselabels=False):
    """Load a Hugin (.net) file into a dict of type {'varname':Multinom(...)}.
    
    By default, uses the node identifiers as variable names; with
    uselabels=True, uses the node labels.
    """
    cpds = huginparser.parseFile(filename)

    # In the Hugin format, the data array is listed such that the child varies fastest, followed by the LAST parent, etc.
    # In the Multinom format, there is no distinction between child/parents; the FIRST variable in the domain varies fastest.
    multinoms = {}
    for var,cpd in cpds.iteritems():
        parents = list(reversed(cpd['parents']))
        if uselabels:
            childname = cpd['label']
            parentnames = [cpds[parent]['label'] for parent in parents]
        else:
            childname = var
            parentnames = parents 
        dom = ( [{childname:cpd['states']}] +
                [{parentname:cpds[parent]['states']} for parent,parentname in zip(parents,parentnames)] )
        par = normalize(map(float,cpd['data']),len(cpd['states']))
        multinoms[childname] = Multinom(dom,par)
    return multinoms
    #TODO: use optimal sharing of domain {domname:domtype}

def loadyaml(filename='tmp.yaml'):
    """Load a YAML file; if it is a Factor, reconstruct it using Factor (sub)class
    constructors.
    """
    with open(filename) as f:
        obj = yaml.safe_load(f.read())
    if isinstance(obj,(Factor,dict)):
        obj = reconstruct(obj)
    return obj

def dumpyaml(tree,filename='tmp.yaml'):
    """Dump a data structure as a YAML file; if it is a factor tree, remove the
    'domlist' and 'domtypes' auxiliary attributes.
    """
    if isinstance(tree,(Factor,dict)):
        tree = remove_doms(tree)
    with open(filename,'w') as f:
        f.write(yaml.dump(tree))

def dumpc(tree,filename='tmp.c'):
    with open(filename,'w') as f:
        f.write(cgenerator.cgen(tree))

def evalc(tree,compiler='unix'):
    """Evaluate a factor by generating C, compiling and executing.
    It uses the gcc compiler provided by distutils.ccompiler.new_compiler.
    Use compiler='msvc' for Microsoft Visual Studio, if you have it."""
    tempdir = tempfile.mkdtemp()
    cfilename = 'tmp.c'
    binfilename = './tmp.out'
    wd = os.getcwd()
    try:
        os.chdir(tempdir)
        dumpc(tree,cfilename)
        cc = distutils.ccompiler.new_compiler(compiler=compiler)
        ofilenames = cc.compile([cfilename])
        cc.link_executable(ofilenames,binfilename)
        lines = subprocess.check_output(binfilename).splitlines()
        if isinstance(tree,Factor):
            result = Multinom(dom=makedom(tree.domlist,tree.domtypes),par=eval(lines[0]))
        else:
            result = setfac({},[ Multinom(dom=makedom(ch.domlist,ch.domtypes),par=eval(line))
                                for ch,line in zip(getfac(tree),lines) ])            
    finally:
        try: os.remove(cfilename)
        except: pass
        try: os.remove(ofilenames[0])
        except: pass
        try: os.remove(binfilename)
        except: pass
        os.chdir(wd)
        os.rmdir(tempdir)
    return result

